package com.good.filters.validators;

import com.fasterxml.jackson.databind.ObjectReader;
import com.good.domain.GDP12CertificateRequest;
import com.good.domain.P12CertificateFailure;
import com.good.filters.request.ResettableStreamHttpServletRequest;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;


/**
 * Class provide deserialization of request body into {@link GDP12CertificateRequest}. If data was successfully 
 * deserialized result is set into {@link ValidationChain#setInputData(GDP12CertificateRequest)}. As input stream from
 * request can be read only once, class replaces original {@link javax.servlet.http.HttpServletRequest} with 
 * {@link ResettableStreamHttpServletRequest} which each time return InputStream which can be read from beginning.
 * 
 * @author Ivan Bilyi
 * 28.10.2015.
 */
public class ObjectMappingValidator implements ValidatorBean {
    private static final Log LOG = LogFactory.getLog(ObjectMappingValidator.class);

    /**
     * Class that deserialize request body from JSON into {@link GDP12CertificateRequest}
     */
    private ObjectReader objectReader;

    /**
     * Checks if data from request body can be deserialize into proper java object.
     * 
     * @param validationChain chain of filters 
     */
    @Override
    public void filter(ValidationChain validationChain) {
        LOG.info("Validating request body.");
        ResettableStreamHttpServletRequest wrappedRequest = new ResettableStreamHttpServletRequest(validationChain.getServletRequest());
        validationChain.setServletRequest(wrappedRequest);
        try {
            GDP12CertificateRequest data = objectReader.readValue(wrappedRequest.getInputStream());
            validationChain.setInputData(data);
        } catch (Exception e) {
            LOG.debug("Badly formatted request. " + e.getMessage());
            validationChain.addErrorMessage(P12CertificateFailure.BAD_REQUEST);            
            return;
        }
        LOG.debug("Request body is valid.");
        validationChain.doValidation();
    }

    /**
     * Set object reader(JSON deserializer)
     */
    @Autowired
    public void setObjectReader(ObjectReader objectReader) {
        this.objectReader = objectReader;
    }
}
