package com.good.filters.request;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.File;

/**
 * Wrapper for {@link HttpServletRequest} which allow to change request path. Class remove all request parameters and
 * add additional values to the path. For example :  original path was {code /pki/handler/?operation=xxx} and 
 * {@link RedirectedHttpServletRequest} was created with {@link RedirectedHttpServletRequest#redirectionPath} that 
 * equals "newOperation" that path that will return thic class will be {@code /pki/handler/newOperation}. 
 * 
 * @author Ivan Bilyi
 * 30.10.2015.
 */
public class RedirectedHttpServletRequest extends HttpServletRequestWrapper {

    /**
     * Path to redirect
     */
    private final String redirectionPath;

    /**
     * Instantiate new HTTP servlet request
     * 
     * @param request the request
     * @param redirectionPath web endpoint path to redirect
     */
    public RedirectedHttpServletRequest(HttpServletRequest request, String redirectionPath) {
        super(request);
        this.redirectionPath = redirectionPath;
    }

    /**
     * @return URL from the request
     */
    @Override
    public StringBuffer getRequestURL() {
        final StringBuffer originalUrl = ((HttpServletRequest) getRequest()).getRequestURL();
        return originalUrl;
    }

    /**
     * @return modified path with new web endpoint
     */
    @Override
    public String getPathInfo() {
        String oldPath = super.getPathInfo() == null ? "" : super.getPathInfo();
        return oldPath + "/" + this.redirectionPath;
    }

    /**
     * @return modified translated path with new web endpoint
     */
    @Override
    public String getPathTranslated() {
        return super.getPathTranslated() + File.separator + this.redirectionPath;
    }

    /**
     * @return empty string, so all original query strings will not be used.
     */
    @Override
    public String getQueryString() {
        return "";
    }

    /**
     * @return modified request URI with new web endpoint
     */
    @Override
    public String getRequestURI() {
        return super.getRequestURI() + "/" + this.redirectionPath;
    }
}
