package com.good.adapters.entrust.impl;

import com.good.SecureRandomStringUtil;
import com.good.adapters.CertificateServer;
import com.good.adapters.entrust.domain.P12CreateRenewRequestData;
import com.good.adapters.entrust.stubs.*;
import com.good.domain.P12CertificateData;
import com.good.exception.CertificateServerException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.PropertySource;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import sun.misc.BASE64Decoder;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.xml.rpc.ServiceException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Entrust MDM WS certificate server adapter. Provide functionality for:
 * <ul>
 * <li>Creation of Certificates. {@link CertificateServer#createCertificate(String, String, String, String, String)}</li>
 * </ul>
 *
 * @author Stanislav Kyfenko
 *         04.08.2015
 */
@PropertySource(value = "classpath:entrustMdmWs.properties")
@Component(value = "ENTRUST")
public class EntrustCertificateServerAdapter implements CertificateServer {
    private static final Log LOG = LogFactory.getLog(EntrustCertificateServerAdapter.class);
    /**
     * Period of time in which keep alive method should be called. Entrust Service will be disconnected after 20 minutes
     * so to be sure that connection will not be broken 15 minutes was chosen as a period.
     */
    private static final long PERIOD = 15 * 60 * 1000;
    /**
     * To make work of Entrust faster "iggroup", "deviceType", "deviceId" parameters are generated randomly. This
     * constant limits size of those strings.
     */
    private static final int STRING_SIZE = 10;

    /**
     * How many times we need to make login when keepAlive() failed
     */
    private static final int REPEAT_TIMES = 5;

    /**
     * URL of Entrust service
     */
    @Value("${entrust.service.url}")
    private String serviceUrl;
    /**
     * Username that is required for login to Entrust
     */
    @Value("${entrust.login}")
    private String login;
    /**
     * Password for Entrust
     */
    @Value("${entrust.password}")
    private String password;

    /**
     * RDN Variables that will be used to generate the DN from the DNFormat
     */
    @Value("${entrust.iggroup}")
    private String iggroup;

    @Value("${entrust.deviceType}")
    private String deviceType;

    @Value("${entrust.deviceId}")
    private String deviceId;

    /**
     * Shows if P12 certificates history will be uploaded
     */
    @Value("${entrust.P12LatestKeyOnly}")
    private String p12LatestKeyOnly;

    /**
     * Digital ID config name
     */
    @Value("${entrust.digital.id.config.name}")
    private String digitalIdConfigName;

    private AdminServiceBindingStub stub;

    /**
     * Method initialize connection to Entrust service
     *
     * @throws CertificateServerException is thrown in case of exception in the remote service during login operation.
     */
    @PostConstruct
    private void initializeStubAndLogin() throws CertificateServerException {
        initializeStub();
        login();
    }

    private void initializeStub() {
        try {
            LOG.info("Initializing stub...");
            URL adminServiceUrl = new URL(serviceUrl);
            AdminService_ServiceLocator locator = new AdminService_ServiceLocator();
            stub = (AdminServiceBindingStub) locator.getAdminService(adminServiceUrl);
            stub.setMaintainSession(Boolean.TRUE);
            LOG.info("Stub was initialized successfully");
        } catch (MalformedURLException | ServiceException e) {
            LOG.error("Failure while initializing stub: " + e);
        }
    }

    /**
     * Method provide login to remote Entrust service. Should be called before any other calls
     *
     * @throws CertificateServerException is thrown in case of exception in the remote service.
     */
    private LoginResult login() throws CertificateServerException {
        LOG.info("Try to login to entrust.");
        LoginCallParms loginCallParms = new LoginCallParms();
        LoginParms parms = new LoginParms();
        parms.setAdminId(login);
        parms.setPassword(decrypt(password));
        loginCallParms.setParms(parms);
        LoginResult result;
        try {
            result = stub.login(loginCallParms);
        } catch (RemoteException e) {
            LOG.error("Login to Entrust failed " + e);
            throw new CertificateServerException("Login to Entrust failed", e);
        }
        if (result != null) {
            if (LoginState.COMPLETE.equals(result.getState())) {
                LOG.info("Login to Entrust was successful");
            } else {
                LOG.info("Login to Entrust was unsuccessful. State of login : " + result.getState());
            }
        }
        return result;
    }

    /**
     * Connection with Entrust service will be interrupted after 20 minutes. To stay connected server should call
     * {@link AdminServiceBindingStub#keepAlive(NameValue[])} with empty array as input parameter. Method are called
     * with period that are regulated by constant value : {@link EntrustCertificateServerAdapter#PERIOD}.
     *
     * @throws CertificateServerException can be thrown when some exception in remote service is thrown during call of
     *                                    {@link AdminServiceBindingStub#keepAlive(NameValue[])}.
     */
    @Scheduled(fixedRate = PERIOD)
    private void keepAlive() throws CertificateServerException {
        LOG.info("Keeping stub alive");
        try {
            stub.keepAlive(new NameValue[0]);
        } catch (RemoteException e) {
            //init counter
            AtomicInteger counter = new AtomicInteger(1);
            // make login as many times as we define in REPEAT_TIMES variable
            while (counter.getAndIncrement() <= REPEAT_TIMES) {
                try {
                    LOG.trace("try to login due to keepAlive was failed. Attempt #" + counter.get());
                    LoginResult loginResult = login();
                    LOG.trace("Got result after login");
                    if (loginResult != null && LoginState.COMPLETE.equals(loginResult.getState())) {
                        LOG.debug("Reconnecting to Entrust was successful with attempt #" + counter.get());
                        // reset counter
                        counter.set(0);
                        return;
                    } else {
                        throw new CertificateServerException("Login to Entrust wasn't successful. LoginState = " + (loginResult != null ? loginResult.getState() : null));
                    }
                } catch (CertificateServerException e1) {
                    LOG.trace("Attempt #" + counter.get() + " was failed", e);
                }
            }
            counter.set(0);
            throw new CertificateServerException("Keep Alive in Entrust failed", e);
        }
        LOG.info("Stub is alive");
    }

    /**
     * Before destroing {@link CertificateServer}  it should logout from Entrust service.
     *
     * @throws CertificateServerException is thrown when exception occurs in the remote service.
     */
    @PreDestroy
    private void logout() throws CertificateServerException {
        try {
            LOG.info("Try to logout");
            stub.logout();
            LOG.info("Logout was successful");
        } catch (RemoteException e) {
            LOG.error("Logout failed " + e);
            throw new CertificateServerException("Logout failed", e);
        }
    }

    /**
     * Digital Id Configuration is used in all operations with Entrust server, such as creation of certificate, renew
     * of certificate or revocation of all certificates.
     *
     * @return digital Id Config from Entrust
     * @throws CertificateServerException is thrown when exception occurs in the remote service.
     */

    private DigitalIdConfigInfo getDigitalIdConfig() throws CertificateServerException {
        DigitalIdConfigGetCallParms callParms = new DigitalIdConfigGetCallParms();
        callParms.setName(digitalIdConfigName);
        LOG.debug("Get digital configuration with name : " + digitalIdConfigName);
        DigitalIdConfigInfo configInfo;
        try {
            configInfo = stub.digitalIdConfigGet(callParms);
        } catch (RemoteException e) {
            LOG.error("getDigitalIdConfig failure" + e);
            throw new CertificateServerException("getDigitalIdConfig failure", e);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received digital configuration : " + configInfo);
        }
        return configInfo;
    }

    /**
     * Create or renew Entrust certificate based on input parameters in requestData
     *
     * @param requestData the request parameters
     * @return P12Certificate
     * @throws CertificateServerException is thrown when exception occurs in the remote service.
     */
    private P12CertificateData createOrRenewCertificate(P12CreateRenewRequestData requestData) throws CertificateServerException {
        UserDigitalIdCreateRecoverCallParms callParms = new UserDigitalIdCreateRecoverCallParms();
        callParms.setUserid(requestData.getUserName());
        UserDigitalIdParms parms = new UserDigitalIdParms();
        callParms.setParms(parms);
        String deviceType = this.deviceType;
        String iggroup = this.iggroup;
        String deviceId = this.deviceId;

        if (StringUtils.isBlank(iggroup)) {
            iggroup = SecureRandomStringUtil.generateRandomStr(STRING_SIZE);
        }

        if (StringUtils.isBlank(deviceType)) {
            if (StringUtils.isNotEmpty(requestData.getDeviceName())) {
                deviceType = requestData.getDeviceName();
            } else {
                deviceType = SecureRandomStringUtil.generateRandomStr(STRING_SIZE);
            }
        }

        if (StringUtils.isBlank(deviceId)) {
            if (StringUtils.isNotEmpty(requestData.getDeviceId())) {
                deviceId = requestData.getDeviceId();
            } else {
                deviceId = SecureRandomStringUtil.generateRandomStr(STRING_SIZE);
            }
        }

        // Configure RDN Variables
        NameValue[] rdnVariables = new NameValue[4];
        rdnVariables[0] = new NameValue("igusername", requestData.getUserName());
        rdnVariables[1] = new NameValue("iggroup", iggroup);
        rdnVariables[2] = new NameValue("devicetype", deviceType);
        rdnVariables[3] = new NameValue("deviceid", deviceId);

        // Configure SubAltNames
        SubjectAltName[] sans = new SubjectAltName[1];
        sans[0] = new SubjectAltName(SubjectAltNameType.EMAIL, requestData.getEmail());

        parms.setDigitalIdConfig(requestData.getDigitalIdConfigInfo().getName());
        parms.setDNVariables(rdnVariables);
        parms.setSecurityStoreType(DigitalIdSecurityStoreType.P12);
        parms.setSecurityStorePassword(requestData.getCertificatePassword());
        parms.setP12LatestKeyOnly(Boolean.valueOf(p12LatestKeyOnly));
        parms.setClientType(deviceType);
        parms.setCreateIfNotExists(requestData.isCreateIfNotExists());
        parms.setRecoverIfExists(requestData.isRecoverIfExists());
        parms.setSubjectAltName(sans);

        P12CertificateData certificateData;
        try {
            LOG.debug("Calling SOAP userDigitalIdCreateRecover call parameters : iggroup='" + iggroup
                    + "', devicetype='" + deviceType + "', deviceid='" + deviceId + '\'');
            LOG.debug("Calling SOAP userDigitalIdCreateRecover method with parameter : " + callParms.toString());
            UserDigitalIdCreateRecoverResult result = stub.userDigitalIdCreateRecover(callParms);
            LOG.debug("SOAP call for certificate creation ended successfully. Result : " + result.toString());
            certificateData = new P12CertificateData(result.getSecurityStore(),
                    P12CertificateData.CertificateTypeEnum.PKCS12);
        } catch (RemoteException e) {
            LOG.error("Couldn't call userDigitalIdCreateRecover SOAP operation", e);
            throw new CertificateServerException("Couldn't call userDigitalIdCreateRecover SOAP operation", e);
        }
        return certificateData;
    }

    /**
     * Forming request data for certificate creation and passing it to {@link EntrustCertificateServerAdapter#createOrRenewCertificate(P12CreateRenewRequestData)}
     *
     * @param userName            - is needed to identify user in entrust.
     * @param email               - is needed to identify user in entrust.
     * @param certificatePassword password to keystore which will be created.
     * @param deviceId            device ID
     * @param deviceName          device name
     * @throws CertificateServerException is thrown when exception occurs in the remote service.
     */
    @Override
    public P12CertificateData createCertificate(String userName, String email, String certificatePassword, String deviceId, String deviceName) throws CertificateServerException {
        DigitalIdConfigInfo digitalIdConfig = getDigitalIdConfig();
        P12CreateRenewRequestData requestData = new P12CreateRenewRequestData.Builder(userName)
                .email(email)
                .digitalIdConfigInfo(digitalIdConfig)
                .certificatePassword(certificatePassword)
                .createIfNotExists(Boolean.TRUE)
                .recoverIfExists(Boolean.TRUE)
                .deviceId(deviceId)
                .deviceName(deviceName)
                .build();
        LOG.debug("Creating certificate for : " + requestData.toString());
        return createOrRenewCertificate(requestData);
    }

    private String decrypt(String inputStr) {
        String decryptedStr = null;
        try {
            decryptedStr = new String(new BASE64Decoder().decodeBuffer(inputStr));
        } catch (IOException e) {
            LOG.error("Decryption error: " + e);
        }
        return decryptedStr;
    }

}
