package com.good.security.x509;

import com.good.domain.P12CertificateFailure;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.WebAttributes;
import org.springframework.web.filter.GenericFilterBean;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.security.cert.X509Certificate;

/**
 * Class provide analysis of requests. If request doesn't contain X509 certificate in it's header attributes class sets
 * {@link WebAttributes#AUTHENTICATION_EXCEPTION} into request attributes. If authentication succeeded 
 * {@link Authentication} will be set into {@link org.springframework.security.core.context.SecurityContext SecurityContext}
 * 
 * @author Stanislav Kyfenko
 *         28.09.2015
 */
public class X509CustomFilter extends GenericFilterBean {

    private static final String X509 = "javax.servlet.request.X509Certificate";

    private AuthenticationManager authenticationManager;

    /**
     * Check if request contains X509 authentication method marker in its header. For such requests invokes {@link X509CustomFilter#doAuthenticate}
     *
     * @param request the request
     * @param response the response
     * @param chain the filter chain
     * @throws IOException may occur wile reading data from servlet request
     * @throws ServletException may occur in one of the filters that will be called in filter chain during call 
     * of {@link FilterChain#doFilter(ServletRequest, ServletResponse)}. 
     */
    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        logger.info("X509CustomFilter");
        if (!(request instanceof HttpServletRequest && response instanceof HttpServletResponse)) {
            logger.debug("Request or response of wrong type, skip from execution.");
            chain.doFilter(request, response);
            return;
        }
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (request.getAttribute(X509) == null && authentication == null) {
            unsuccessfulAuthentication((HttpServletRequest) request, (HttpServletResponse) response, new BadCredentialsException("No X509 certificate"));
            throw new AccessDeniedException(P12CertificateFailure.UNKNOWN_CERT.toString());
        } else if (authentication != null && authentication.getPrincipal() != null) {
            if(logger.isDebugEnabled()) {
                logger.debug("No certificate but user already authenticated." + authentication);
            }
            chain.doFilter(request, response);
            return;
        }

        X509Certificate[] certificates = (X509Certificate[]) request.getAttribute(X509);
        if (certificates.length > 0) {
            doAuthenticate((HttpServletRequest) request, (HttpServletResponse) response, certificates[0]);
        }

        chain.doFilter(request, response);
    }

    /**
     * Perform authentication of user using {@link X509Certificate} as parameter of authentication.
     * Authentication will be performed as follows: 
     * <ul>
     *     <li>{@link AuthenticationManager#authenticate(Authentication)} method is executed.</li>
     *     <li>In case of no exceptions will be thrown by {@link AuthenticationManager} result of it's work will be 
     *     store in {@link org.springframework.security.core.context.SecurityContext#setAuthentication(Authentication) SecurityContext.setAuthentication()}</li>
     *     <li>In case of {@link AuthenticationException} {@link WebAttributes#AUTHENTICATION_EXCEPTION} will be set into servlet request.</li>
     * </ul>
     * @param request the request
     * @param response the response
     * @param certificate X509 certificate, which should be used to authenticate user
     */
    private void doAuthenticate(HttpServletRequest request, HttpServletResponse response, X509Certificate certificate) {
        Authentication authResult;

        if (certificate == null) {
            logger.debug("No certificate found in request");
            return;
        }

        if (logger.isDebugEnabled()) {
            logger.debug("preAuthenticatedPrincipal = " + certificate + ", trying to authenticate");
        }

        try {
            X509AuthenticationToken authRequest = new X509AuthenticationToken(certificate, getPreAuthenticatedCredentials(request));
            authResult = authenticationManager.authenticate(authRequest);
            successfulAuthentication(request, response, authResult);
        } catch (AuthenticationException failed) {
            unsuccessfulAuthentication(request, response, failed);
            throw failed;
        }
    }

    /**
     * Sets authentication manager.
     */
    public void setAuthenticationManager(AuthenticationManager authenticationManager) {
        this.authenticationManager = authenticationManager;
    }

    /**
     * Gets pre authenticated credentials.
     *
     * @param request the request
     * @return the pre authenticated credentials
     * @see org.springframework.security.web.authentication.preauth.AbstractPreAuthenticatedProcessingFilter#getPreAuthenticatedPrincipal(javax.servlet.http.HttpServletRequest)
     */

    protected Object getPreAuthenticatedCredentials(HttpServletRequest request) {
        return "N/A";
    }

    /**
     * Unsuccessful authentication. Clearing context and setting error to the request
     * @param request the request
     * @param response the response
     * @param failed an exception that occurred
     */
    protected void unsuccessfulAuthentication(HttpServletRequest request, HttpServletResponse response, AuthenticationException failed) {
        SecurityContextHolder.clearContext();

        logger.debug("Cleared security context due to exception", failed);

        request.setAttribute(WebAttributes.AUTHENTICATION_EXCEPTION, failed);
    }

    /**
     * Puts the <code>Authentication</code> instance returned by the authentication manager into the secure context.
     */
    protected void successfulAuthentication(HttpServletRequest request, HttpServletResponse response, Authentication authResult) {
        if (logger.isDebugEnabled()) {
            logger.debug("Authentication success: " + authResult);
        }
        SecurityContextHolder.getContext().setAuthentication(authResult);
    }
}
