package com.good.filters.validators;

import com.good.domain.GDP12CertificateRequest;
import com.good.domain.GenericData;
import com.good.domain.P12CertificateData;
import com.good.domain.P12CertificateFailure;
import com.good.filters.utils.DataSerializer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.*;

/**
 * Class that handles validation of requests. Class contains all required information for validation of response. 
 * Validation is performed by instances of {@link ValidatorBean} which are stored in {@link Queue}. In each iteration 
 * new realization of {@link ValidatorBean} is polled from {@link Queue} while there will be no validators in it.
 * 
 * @author Ivan Bilyi
 * 28.10.2015.
 */
public class ValidationChain {

    private static final Log LOG = LogFactory.getLog(ValidationChain.class);
    /**
     * Queue of filters that will perform validation of requests. In each call of {@link ValidationChain#doValidation()} new object is taken from Queue and 
     */
    private Queue<ValidatorBean> filterBeans;
    /**
     * Servlet request
     */
    private HttpServletRequest servletRequest;
    /**
     * Servlet response
     */
    private HttpServletResponse servletResponse;
    /**
     * Deserialize data from servlet request body
     */
    private GDP12CertificateRequest inputData;
    /**
     * Bean that is responsible for serialization of {@link com.good.domain.P12CertificateData} into JSON
     */
    private DataSerializer dataSerializer;
    /**
     * indicates that during work of filters all data in request was valid 
     */
    private boolean allDataValid = true;

    /**
     * Instantiates Validation chain
     * 
     * @param builder object builder
     */
    private ValidationChain(Builder builder) {
        this.filterBeans = builder.getFilterBeans();
        this.servletRequest = builder.getServletRequest();
        this.servletResponse = builder.getServletResponse();
        this.dataSerializer = builder.getDataSerializer();
    }

    /**
     * validate request data using next filter in {@link Queue}
     */
    public void doValidation() {
        ValidatorBean currentFilter = filterBeans.poll();
        if(currentFilter != null) {
            currentFilter.filter(this);
        }
    }

    /**
     * @return the request
     */
    public HttpServletRequest getServletRequest() {
        return servletRequest;
    }

    /**
     * Set servlet request
     */
    public void setServletRequest(HttpServletRequest servletRequest) {
        this.servletRequest = servletRequest;
    }

    /**
     * @return the response
     */
    public HttpServletResponse getServletResponse() {
        return servletResponse;
    }

    /**
     * Set servlet response
     */
    public void setServletResponse(HttpServletResponse servletResponse) {
        this.servletResponse = servletResponse;
    }

    /**
     * @return deserialize request data. This field is set only after {@link ObjectMappingValidator} class work.
     */
    public GDP12CertificateRequest getInputData() {
        return inputData;
    }

    /**
     * Set deserialize request data
     */
    public void setInputData(GDP12CertificateRequest inputData) {
        this.inputData = inputData;
    }

    /**
     * true - in case of no errors was found during request validation
     */
    public boolean isAllDataValid() {
        return allDataValid;
    }

    /**
     * Add response body with failure reason to servlet response 
     * 
     * @param failureReason response body to serialize
     */
    public void addErrorMessage(P12CertificateFailure failureReason) {
        String reqId = null;
        if (inputData != null) {
            reqId = inputData.getReqId();
        }
        LOG.debug("Found error during validation : " + failureReason);
        dataSerializer.writeResponse(new GenericData(failureReason, reqId), servletResponse);
        this.allDataValid = false;
    }

    public static class Builder {

        private Queue<ValidatorBean> filterBeans;
        private HttpServletRequest servletRequest;
        private HttpServletResponse servletResponse;
        private DataSerializer dataSerializer;
        
        public Builder(){}
        public ValidationChain build(){
            if(filterBeans == null || servletRequest == null || servletResponse == null) {
                throw new IllegalArgumentException("Some of mandatory fields was not set up.");
            }
            return new ValidationChain(this);
        }
        public Builder setFilterBeans(List<ValidatorBean> filterBeans) {
            this.filterBeans = new ArrayDeque<>(filterBeans);
            return this;
        }

        public Builder setServletRequest(HttpServletRequest servletRequest) {
            this.servletRequest = servletRequest;
            return this;
        }

        public Builder setServletResponse(HttpServletResponse servletResponse) {
            this.servletResponse = servletResponse;
            return this;
        }

        public Builder setDataSerializer(DataSerializer dataSerializer) {
            this.dataSerializer = dataSerializer;
            return this;
        }

        public DataSerializer getDataSerializer() {
            return dataSerializer;
        }

        public HttpServletResponse getServletResponse() {
            return servletResponse;
        }

        public HttpServletRequest getServletRequest() {
            return servletRequest;
        }

        public Queue<ValidatorBean> getFilterBeans() {
            return filterBeans;
        }
    }

}
