package com.good.filters.validators;

import com.good.config.SupportedOperations;
import com.good.domain.P12CertificateFailure;
import com.good.filters.request.RedirectedHttpServletRequest;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.servlet.http.HttpServletRequest;

/**
 * Certificate Management Protocol requires support different operations using same web endpoint and request parameters.
 * This filter transforms request web endpoint from format {@code /pki/?operation=xxx} into {@code /pki/xxx} depending 
 * on value of  request parameter and authentication method that was specified in request header.
 * To change request path {@link RedirectedHttpServletRequest} is used. Class changes servlet request in {@link ValidationChain}
 * 
 * @author Ivan Bilyi
 * 30.10.2015.
 */
public class PathModificationValidator implements ValidatorBean {

    private static final String OPERATION = "operation";
    private static final String X509 = "javax.servlet.request.X509Certificate";
    private static final Log LOG = LogFactory.getLog(PathModificationValidator.class);

    /**
     * Method modify request web endpoint path into path that can be handled by service.
     * In case of /pki/?operation=xxx method check authentication method and redirect 
     * execution into proper web endpoint in the service.
     * 
     * @param validationChain chain of filters 
     */
    @Override
    public void filter(ValidationChain validationChain) {
        LOG.debug("PathModificationValidator" );
        HttpServletRequest request = validationChain.getServletRequest();
        String path = request.getPathInfo();
        if(StringUtils.isEmpty(path)) {
            final String operationValue = request.getParameter(OPERATION);
            if(SupportedOperations.GET_USER_KEY_PAIR.getValue().equals(operationValue)) {
                String newOperation;
                String authType = request.getHeader("Authorization");
                if(authType != null && authType.startsWith("Basic ")) {
                    newOperation = "createWithBasicAuth";
                } else if(request.getAttribute(X509) != null) {
                    newOperation = "createWithClientAuth";
                } else {
                    LOG.debug("No authentication found, redirection didn't performed.");
                    validationChain.addErrorMessage(P12CertificateFailure.UNKNOWN_REQUEST);
                    return;
                }
                validationChain.setServletRequest(new RedirectedHttpServletRequest(request, newOperation));
                LOG.debug("Redirecting \"?operation=getUserKeyPair\" -> \"/" + newOperation +"\"" );
            } else if (SupportedOperations.contains((operationValue))){
                LOG.debug("Redirecting  -> \"/" + operationValue +"\"" );
                validationChain.setServletRequest(new RedirectedHttpServletRequest(request, operationValue));
            } else {
                LOG.debug("Redirecting doesn't performed, as parameter not supported. Was : " + operationValue);
            }
        } else {
            LOG.debug("Redirecting doesn't performed, as path not empty.");
        }
        validationChain.doValidation();
    }
}
