package com.good.service;

import com.good.SecureRandomStringUtil;
import com.good.adapters.CertificateServer;
import com.good.adapters.CertificateServer.CertificateServerType;
import com.good.adapters.factory.CertificateServerFactory;
import com.good.annotations.SkipBodyValidation;
import com.good.domain.*;
import com.good.exception.CertificateServerException;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.PropertySource;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.ws.rs.core.MediaType;
import java.util.Map;

/**
 * Web endpoint to support several certificate server types like Entrust, Microsoft NDES.
 *
 * @author Stanislav Kyfenko
 *         05.08.2015
 */
@Controller
@PropertySource(value = "classpath:entrustMdmWs.properties")
public class CertificateServerService {
    private static final Log LOG = LogFactory.getLog(CertificateServerService.class);

    /**
     * When request doesn't contains {@link GDP12CertificateRequest#authToken} field this value will be used for
     * limitation of randomly generated password to decrypt certificate from generate PKCS12 keystore.
     */
    private static final int PASSWORD_LENGTH = 15;

    /**
     * Entrust service requires username separate from email, which are not defined at gd-ca-protocol.
     * This value will be used for creation of certificate, not for log in to Entrust service. User name for
     * certificate creation should be specified in entrustMdmWs.properties file.
     */
    @Value("${request.userName}")
    private String userName;

    /**
     * Factory class that should return adapters to various {@link CertificateServer}. Type that is required by each
     * request should be added in request body, but currently not supported by gd-ca-protocol. So only
     * {@link CertificateServer.CertificateServerType#ENTRUST} can be used as server type.
     */
    @Autowired
    private CertificateServerFactory certificateServerFactory;

    /**
     * Web endpoint for testing server availability of remote host.
     *
     * @return json message
     */
    @RequestMapping(value = "/test", method = RequestMethod.GET, produces = MediaType.APPLICATION_JSON)
    @ResponseStatus(HttpStatus.OK)
    public
    @ResponseBody
    Map<String, String> test() {
        return new ImmutableMap.Builder<String, String>().put("message", "It works!").build();
    }

    /**
     * Web endpoint, which doesn't require any authentication and which should be used for creation of certificates.
     *
     * @param request deserialize request body
     * @return json message that contains PKCS12 certificate
     */
    private GenericData createCertificate(GDP12CertificateRequest request) {
        P12CertificateData certificateData;
        LOG.info("createCertificate : " + request.toString());
        try {
            CertificateServer certificateServer = certificateServerFactory.getInstance(CertificateServerType.ENTRUST);
            String authToken = request.getAuthToken();
            if (StringUtils.isEmpty(authToken)) {
                authToken = SecureRandomStringUtil.generateRandomStr(PASSWORD_LENGTH);
                certificateData = certificateServer.createCertificate(userName, request.getUser(), authToken, request.getDeviceId(), request.getDeviceName());
                certificateData.setPassword(authToken);
            } else {
                certificateData = certificateServer.createCertificate(userName, request.getUser(), authToken, request.getDeviceId(), request.getDeviceName());
            }
            certificateData.setReqId(request.getReqId());
        } catch (CertificateServerException e) {
            LOG.error("Certificate creation ended with exception: " + e);
            certificateData = new P12CertificateData(P12CertificateFailure.UNKNOWN, request.getReqId());
        }
        LOG.info("Certificate was generated successfully : " + certificateData.toString());
        return certificateData;
    }

    /**
     * Web endpoint, which requires basic authentication. It should be used for creation of certificates.
     *
     * @param request deserialize request body
     * @return json message that contains PKCS12 certificate
     */
    @RequestMapping(value = "/createWithBasicAuth", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON, consumes = MediaType.APPLICATION_JSON)
    @ResponseStatus(HttpStatus.OK)
    public
    @ResponseBody
    GenericData createWithBasicAuth(@RequestBody GDP12CertificateRequest request) {
        return createCertificate(request);
    }

    /**
     * Web endpoint, which requires X509 certificate based authentication and which should be used for creation of certificates.
     *
     * @param request deserialize request body
     * @return json message that contains PKCS12 certificate
     */
    @RequestMapping(value = "/createWithClientAuth", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON, consumes = MediaType.APPLICATION_JSON)
    @ResponseStatus(HttpStatus.OK)
    public
    @ResponseBody
    GenericData createWithClientAuth(@RequestBody GDP12CertificateRequest request) {
        return createCertificate(request);
    }

    /**
     * Web endpoint, which returns array of all the commands implemented by the connector server after Basic or X509 certificate based authentication.
     *
     * @return json message that contains array of all the commands implemented by the connector server
     */
    @SkipBodyValidation
    @RequestMapping(value = "/getInfo", method = RequestMethod.GET, produces = MediaType.APPLICATION_JSON)
    @ResponseStatus(HttpStatus.OK)
    public
    @ResponseBody
    GetInfoDataResponse getInfo() {
        return new GetInfoDataResponse();
    }
}
