package com.good.filters.request;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.*;

/**
 * As {@link InputStream} can be read only once this class allow to reset input stream each time 
 * {@link HttpServletRequest#getInputStream()} or {@link HttpServletRequest#getReader()} are called. Class reads content
 * of original {@link InputStream} into array of bytes and return wrapped {@link ServletInputStream} with
 * {@link ByteArrayInputStream} inside it.  
 * 
 * @author Ivan Bilyi
 * 28.10.2015.
 */
public class ResettableStreamHttpServletRequest extends HttpServletRequestWrapper {
    private static final Log LOG = LogFactory.getLog(ResettableStreamHttpServletRequest.class);

    /**
     * Servlet request body
     */
    private byte[] rawData;
    /**
     * Original servlet request
     */
    private HttpServletRequest request;
    /**
     * Input stream that can be reset
     */
    private ResettableServletInputStream servletStream;

    /**
     * Instantiate new HTTP servlet request
     * 
     * @param request the original HTTP servlet request 
     */
    public ResettableStreamHttpServletRequest(HttpServletRequest request) {
        super(request);
        this.request = request;
        this.servletStream = new ResettableServletInputStream();
    }

    /**
     * @return the input stream which was resetted, so it can be read again
     * @throws IOException I/O exception
     */
    @Override
    public ServletInputStream getInputStream() throws IOException {
        initInputStream();
        return servletStream;
    }

    /**
     * @return the input stream which was resetted, so it can be read again
     * @throws IOException I/O exception
     */
    @Override
    public BufferedReader getReader() throws IOException {
        initInputStream();
        return new BufferedReader(new InputStreamReader(servletStream));
    }

    /**
     * Reads data from original servlet request and resets input stream to be able to read it again
     * @throws IOException I/O exception
     */
    private void initInputStream() throws IOException {
        if (rawData == null) {
            rawData = IOUtils.toByteArray(this.request.getInputStream());
        }
        servletStream.stream = new ByteArrayInputStream(rawData);
    }

    /**
     * Input stream that can be resetted
     */
    private static class ResettableServletInputStream extends ServletInputStream {

        private InputStream stream;

        @Override
        public int read() throws IOException {
            return stream.read();
        }
    }
}
