package com.good.adapters.entrust.domain;

import com.good.adapters.entrust.stubs.DigitalIdConfigInfo;

/**
 * Data used as input to create-renew the certificate
 *
 * @author Stanislav Kyfenko
 *         04.08.2015
 */
public class P12CreateRenewRequestData {

    /**
     * Is used as call parameter for creation and renew of certificate.
     */
    private String userName;
    /**
     * Is used in array of {@link com.good.adapters.entrust.stubs.SubjectAltName} values to add to the digital Ids. 
     * Supported only for use with an  Entrust CA. With a Microsoft CA this setting is ignored (has no effect).
     */
    private String email;
    /**
     * Password with which keystore will be created. Using this password client will be able to decrypt keystore and 
     * get Certificate.
     */
    private String certificatePassword;
    /**
     * Information about a Digital Id Config.
     */
    private DigitalIdConfigInfo digitalIdConfigInfo;
    /**
     * A flag indicating if new certificate should be created in case it was not created before
     */
    private boolean createIfNotExists;
    /**
     * A flag indicating if existing certificate should be recovered
     */
    private boolean recoverIfExists;
    /**
     * Device ID
     */
    private String deviceId;
    /**
     * Device name
     */
    private String deviceName;

    private P12CreateRenewRequestData(Builder builder) {
        this.userName = builder.userName;
        this.email = builder.email;
        this.certificatePassword = builder.certificatePassword;
        this.digitalIdConfigInfo = builder.digitalIdConfigInfo;
        this.createIfNotExists = builder.createIfNotExists;
        this.recoverIfExists = builder.recoverIfExists;
        this.deviceId = builder.deviceId;
        this.deviceName = builder.deviceName;
    }
    
    public String toString() {
        StringBuilder sb = new StringBuilder("[ userName:");
        sb.append(userName)
                .append(", email:").append(email)
                .append(", digitalIdConfigInfo:").append(digitalIdConfigInfo.getName())
                .append(", createIfNotExists:").append(createIfNotExists)
                .append(", recoverIfExists:").append(recoverIfExists)
                .append(", deviceId:").append(deviceId)
                .append(", deviceName:").append(deviceName)
                .append("]");
        return sb.toString();
    }

    public static class Builder {
        private final String userName;
        private String email;
        private String certificatePassword;
        private DigitalIdConfigInfo digitalIdConfigInfo;
        private boolean createIfNotExists;
        private boolean recoverIfExists;
        private String deviceId;
        private String deviceName;

        public Builder(String userName) {
            this.userName = userName;
        }

        public Builder email(String email) {
            this.email = email;
            return this;
        }

        public Builder certificatePassword(String certificatePassword) {
            this.certificatePassword = certificatePassword;
            return this;
        }

        public Builder digitalIdConfigInfo(DigitalIdConfigInfo digitalIdConfigInfo) {
            this.digitalIdConfigInfo = digitalIdConfigInfo;
            return this;
        }

        public Builder createIfNotExists(boolean createIfNotExists) {
            this.createIfNotExists = createIfNotExists;
            return this;
        }

        public Builder recoverIfExists(boolean recoverIfExists) {
            this.recoverIfExists = recoverIfExists;
            return this;
        }

        public Builder deviceId(String deviceId) {
            this.deviceId = deviceId;
            return this;
        }

        public Builder deviceName(String deviceName) {
            this.deviceName = deviceName;
            return this;
        }

        public P12CreateRenewRequestData build() {
            return new P12CreateRenewRequestData(this);
        }
    }

    public String getUserName() {
        return userName;
    }

    public String getEmail() {
        return email;
    }

    public String getCertificatePassword() {
        return certificatePassword;
    }

    public DigitalIdConfigInfo getDigitalIdConfigInfo() {
        return digitalIdConfigInfo;
    }

    public boolean isCreateIfNotExists() {
        return createIfNotExists;
    }

    public boolean isRecoverIfExists() {
        return recoverIfExists;
    }

    public String getDeviceId() {
        return deviceId;
    }

    public String getDeviceName() {
        return deviceName;
    }
}
