package com.good.filters.validators;

import com.good.annotations.SkipBodyValidation;
import com.good.domain.P12CertificateFailure;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import java.lang.reflect.Method;
import java.util.*;

/**
 * Validation of request web endpoint. Method check if controller has handler for endpoint specified in servlet request.
 * If no handler will be found, will be returned response with {@link P12CertificateFailure#UNKNOWN_REQUEST} error code
 * in it's body
 * 
 * @author Ivan Bilyi
 * 28.10.2015.
 */
public class RequestValidator implements ValidatorBean {
    private static final Log LOG = LogFactory.getLog(RequestValidator.class);

    /**
     * Map of web endpoint path and it's mapping information
     */
    private Map<String, RequestMappingInfo> pathMapping;
    private Map<String, Method> annotationPresentForMethod;

    /**
     * Validation of request web endpoint. 
     * Method check if service has handler for endpoint specified in servlet request
     * 
     * @param validationChain chain of filters that should be proceeded next
     */
    @Override
    public void filter(ValidationChain validationChain) {
        LOG.info("Validating request.");
        String pathInfo = validationChain.getServletRequest().getPathInfo();
        RequestMappingInfo info = pathMapping.get(pathInfo);
        if(info != null) {
            RequestMappingInfo matchingInfo = info.getMatchingCondition(validationChain.getServletRequest());
            if(matchingInfo != null) {
                LOG.debug("Request path is valid. Handler was found.");
                if (!annotationPresentForMethod.get(pathInfo).isAnnotationPresent(SkipBodyValidation.class)) {
                    validationChain.doValidation();
                }
                return;
            }
        }
        LOG.debug("There is no handler method for request " + pathInfo);
        validationChain.addErrorMessage(P12CertificateFailure.UNKNOWN_REQUEST);
    }

    /**
     * Method generate map of web endpoint path to request information that can be handled in this web endpoint.
     * 
     * @param handlerMapping mapping of request information into proper handler method
     */
    @Autowired
    public void setHandlerMapping(RequestMappingHandlerMapping handlerMapping) {
        Map<RequestMappingInfo, HandlerMethod> handlerMethods = handlerMapping.getHandlerMethods();
        this.pathMapping = new HashMap<>();
        this.annotationPresentForMethod=new HashMap<>();
        handlerMethods.forEach((key,value) -> {

            Set<String> pathValues = key.getPatternsCondition().getPatterns();
            for(String pathValue: pathValues) {
                annotationPresentForMethod.put(pathValue,value.getMethod());
                pathMapping.put(pathValue,key);
            }
        });
    }
}
