package com.good.filters;

import com.good.filters.utils.DataSerializer;
import com.good.filters.validators.ValidationChain;
import com.good.filters.validators.ValidatorBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.filter.GenericFilterBean;

import javax.annotation.Resource;
import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.util.Arrays;
import java.util.List;

/**
 * Class provides validation of requested web endpoint and content of request body. Validation are performed by 
 * {@link ValidatorBean ValidatorBeans}. {@link ValidatorBean ValidatorBeans} should be executed as follows :
 * <ol>
 *     <li>{@link com.good.filters.validators.PathModificationValidator PathModificationValidator} modify request 
 *     endpoint from /pki/?operation=xxx to one that can be handled by 
 *     {@link com.good.service.CertificateServerService CertificateServerService}</li>
 *     <li>{@link com.good.filters.validators.RequestValidator RequestValidator} checks that request path can be 
 *     supported by {@link com.good.service.CertificateServerService CertificateServerService}. If it doesn't supported 
 *     response with proper error code will be returned.</li>
 *     <li>{@link com.good.filters.validators.ObjectMappingValidator ObjectMappingValidator} checks that data in the 
 *     request body can be deserialized into {@link com.good.domain.GDP12CertificateRequest}</li>
 *     <li>{@link com.good.filters.validators.RequestDataValidator RequestDataValidator} provide check that all 
 *     mandatory fields are presented in requests body.</li>
 * </ol>
 * Execution of {@link ValidatorBean ValidatorBeans} are controlled by {@link ValidationChain}. For each request new 
 * instance of {@link ValidationChain} should be created.
 * 
 * @see com.good.domain.P12CertificateFailure
 * @author Ivan Bilyi
 * 25.10.2015.
 */
public class RequestValidationFilter extends GenericFilterBean {
    /**
     * List of filters that will perform validation of requests 
     */
    private List<ValidatorBean> validationFilters;

    /**
     * Bean that is responsible for serialization of {@link com.good.domain.P12CertificateData} into JSON
     */
    @Autowired
    private DataSerializer dataSerializer;

    /**
     * Method provide validation of request in following steps:
     * <ul>
     *     <li>Modification of web endpoint for /pki/operation=xxx</li>
     *     <li>Validation of request web endpoint</li>
     *     <li>Check if request body contains valid data</li>
     *     <li>Check if all mandatory fields are specified in request body</li>
     * </ul>
     * 
     * @param servletRequest the request
     * @param servletResponse the response
     * @param filterChain the filter chain
     */
    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
        logger.info("Validation of request.");
        if (!(servletRequest instanceof HttpServletRequest && servletResponse instanceof HttpServletResponse)) {
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }
        HttpServletRequest request = (HttpServletRequest) servletRequest;
        HttpServletResponse response = (HttpServletResponse) servletResponse;
        ValidationChain validationChain = new ValidationChain.Builder()
                .setFilterBeans(validationFilters)
                .setDataSerializer(dataSerializer)
                .setServletRequest(request)
                .setServletResponse(response)
                .build();
        validationChain.doValidation();
        if(validationChain.isAllDataValid()) {
            logger.debug("All data is valid.");
            filterChain.doFilter(validationChain.getServletRequest(), validationChain.getServletResponse());
        } else {
            logger.debug("Some errors was found.");
        }
    }

    @Resource
    public void setValidationFilters(ValidatorBean[] validationFilters) {
        this.validationFilters = Arrays.asList(validationFilters);
    }
}
